<?php
/**
 * Plugin Name: SiteBlazer AI Content & Image Replacer
 * Description: Automatically replaces {placeholders} in posts, pages, and Elementor content using a remote JSON file, and swaps hardcoded images with high-quality images from Pexels for AI-built websites.
 * Version: 1.0
 * Author: Blazers IO
 */

// --- START: CONFIGURE YOUR SETTINGS HERE ---

// 1. Enter your Pexels API Key.
define('SITEBLAZER_PEXELS_API_KEY', 'BQIToO8wyDpA1LADWEpHXJtwIPVw1bGKFkdBut4Nhrlgn3rGv1agj1c1');

// 2. Enter the image category.
$img_category = get_option('plugin_default_category', 'Wordpress');
$image_category = strtolower($img_category);
define('SITEBLAZER_IMAGE_CATEGORY', $image_category);

// 3. Enter the list of OLD image FILENAMES to replace.
define('SITEBLAZER_OLD_FILENAMES', [
    'Home-page-banner-image.png',
    'Home-page-about-image.png',

    'Home-page-service-image1.png',
    'Home-page-service-image2.png',
    'Home-page-service-image3.png',
    'Home-page-service-image4.jpg',

    'About-page-image1.png',
    'About-page-image2.png',

    'Service-page-image1.png',
    'Service-page-image2.png',
    'Service-page-image3.png',
    'Service-page-image4.png',
]);


// --- END: CONFIGURE YOUR SETTINGS HERE ---

define('SITEBLAZER_IMAGE_SIZES', [
    // Home page
    'Home-page-banner-image.png'   => [626, 626],
    'Home-page-about-image.png'    => [626, 417],

    'Home-page-service-image1.png' => [330, 460],
    'Home-page-service-image2.png' => [330, 460],
    'Home-page-service-image3.png' => [330, 460],
    'Home-page-service-image4.jpg' => [330, 460],

    // About page
    'About-page-image1.png'        => [626, 417],
    'About-page-image2.png'        => [626, 417],

    // Service page
    'Service-page-image1.png'      => [626, 626],
    'Service-page-image2.png'      => [626, 626],
    'Service-page-image3.png'      => [626, 626],
    'Service-page-image4.png'      => [626, 626],
]);


/**
 * The main trigger function that runs the heavy replacement.
 */
function siteblazer_run_on_page_load($force = false) {
    static $is_running = false;
    if ($is_running) return;
    $is_running = true;

    if (!$force && (is_admin() || wp_doing_cron())) return;

    global $wpdb;
    $any_changes_made = false;

    foreach (SITEBLAZER_OLD_FILENAMES as $old_filename) {
        $match_found = false;
        $all_tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);
        foreach ($all_tables as $table) {
            $table_name = $table[0];
            try { $columns = $wpdb->get_results("SHOW COLUMNS FROM `$table_name`"); } catch (Exception $e) { continue; }
            if (!$columns) continue;
            foreach ($columns as $column) {
                $col_name = $column->Field;
                if (strpos(strtolower($column->Type), 'text') !== false) {
                    $count = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$table_name` WHERE `$col_name` LIKE %s", '%' . $wpdb->esc_like($old_filename) . '%'));
                    if ($count > 0) {
                        $match_found = true;
                        break 2;
                    }
                }
            }
        }

        if ($match_found) {
            
			$pexels_url = siteblazer_get_pexels_image(
    SITEBLAZER_PEXELS_API_KEY,
    SITEBLAZER_IMAGE_CATEGORY,
    $old_filename
);

            if (!$pexels_url) continue;

			$upload = siteblazer_download_image($pexels_url, $old_filename);

            if (!$upload) continue;

            $new_full_url = $upload['url'];

            if (!is_admin()) {
                siteblazer_console_log("New Image URL: " . $new_full_url);
            }

            $basename = basename(urldecode(trim($old_filename))); 
            if (strtolower($basename) === 'brand-logo.png') {
                $new_full_url = get_option('logo_image_path', '');
                if (!empty($new_full_url)) {
                    $new_full_url = strtolower($new_full_url);
                    if (siteblazer_safe_database_replace($old_filename, $new_full_url)) {
                        $any_changes_made = true;
                    }
                }
            } else {
                if (siteblazer_safe_database_replace($old_filename, $new_full_url)) {
                    $any_changes_made = true;
                }
            }
        }
    }

    if ($any_changes_made) {
        if (did_action('elementor/loaded')) {
            \Elementor\Plugin::$instance->files_manager->clear_cache();
        }
    }
}


/**
 * ------------------- PLACEHOLDER CONTENT REPLACEMENT -------------------
 * Everything below remains exactly as your working plugin code
 * JSON placeholders, Elementor content, WooCommerce category updates
 */

function json_placeholder_get_json_data() {
    static $json_data = null;
    if ($json_data !== null) return $json_data;

    $category = get_option('plugin_default_category', 'Wordpress');
    $filename = strtolower($category) . '_placeholder_wp.json';
    $category_clean = strtolower(preg_replace('/\s+/', '_', trim($category)));
	$file_path = "https://siteblazer.live/wordpress_placeholder/{$category_clean}_placeholder.json";

    $response = wp_remote_get($file_path, ['timeout' => 15]);
    if (is_wp_error($response)) return null;
    $body = wp_remote_retrieve_body($response);
    $json_data = json_decode($body, true);

    return is_array($json_data) ? $json_data : null;
}

function json_placeholder_replace_text($text, $json_data) {
    return preg_replace_callback('/\{([a-zA-Z0-9_.]+)\}/', function ($matches) use ($json_data) {
        $keys = explode('.', $matches[1]);
        $value = $json_data;
        foreach ($keys as $key) {
            if (!isset($value[$key])) return $matches[0];
            $value = $value[$key];
        }
        return esc_html($value);
    }, $text);
}

function json_placeholder_deep_replace(&$data, $json_data) {
    foreach ($data as $key => &$value) {
        if (is_array($value)) {
            json_placeholder_deep_replace($value, $json_data);
        } elseif (is_string($value) && strpos($value, '{') !== false) {
            $value = json_placeholder_replace_text($value, $json_data);
        }
    }
}

register_activation_hook(__FILE__, 'siteblazer_on_activate');

function siteblazer_on_activate() {

    // Safety: run only once
    if (get_option('siteblazer_replacements_done')) {
        return;
    }

    // Run heavy tasks
    siteblazer_run_on_page_load(true);
    siteblazer_run_json_placeholder_once();
    siteblazer_replace_category_terms_once();

    // Mark completed
    update_option('siteblazer_replacements_done', 1);
}



function siteblazer_run_json_placeholder_once() {

    $json_data = json_placeholder_get_json_data();
    if (!$json_data) return;

    $post_types = ['post', 'page', 'elementor_library', 'product','wp_template_part','wp_template'];
    $query = new WP_Query([
        'post_type'      => $post_types,
        'posts_per_page' => -1,
        'post_status'    => 'any',
    ]);

    foreach ($query->posts as $post) {
        $post_id = $post->ID;

        $title = $post->post_title;
        $new_title = json_placeholder_replace_text($title, $json_data);
        if ($new_title !== $title) {
            wp_update_post(['ID' => $post_id, 'post_title' => $new_title]);
        }

        $content = $post->post_content;
        $new_content = json_placeholder_replace_text($content, $json_data);
        if ($new_content !== $content) {
            wp_update_post(['ID' => $post_id, 'post_content' => $new_content]);
        }

        $excerpt = $post->post_excerpt;
        $new_excerpt = json_placeholder_replace_text($excerpt, $json_data);
        if ($new_excerpt !== $excerpt) {
            wp_update_post(['ID' => $post_id, 'post_excerpt' => $new_excerpt]);
        }

        $elementor_data_raw = get_post_meta($post_id, '_elementor_data', true);
        if ($elementor_data_raw) {
            $data = json_decode($elementor_data_raw, true);
            if (is_array($data)) {
                json_placeholder_deep_replace($data, $json_data);
                update_post_meta($post_id, '_elementor_data', wp_slash(json_encode($data)));
            }
        }
    }
}

/**
 * ------------------- HELPER FUNCTIONS -------------------
 * Keep all your helper functions as-is
 */

function siteblazer_safe_database_replace($old_filename, $new_url) {
    global $wpdb;
    $changes_were_made = false;
    
    $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);
    foreach ($tables as $table) {
        $table_name = $table[0];
        try { $columns = $wpdb->get_results("SHOW COLUMNS FROM `$table_name`"); } catch (Exception $e) { continue; }
        if (!$columns) continue;

        $primary_key = null;
        foreach($columns as $column) { if ($column->Key == 'PRI') { $primary_key = $column->Field; break; } }
        if (!$primary_key) continue;

        foreach ($columns as $column) {
            $col_name = $column->Field;
            if (strpos(strtolower($column->Type), 'text') !== false) {
                
                $rows = $wpdb->get_results($wpdb->prepare("SELECT `{$primary_key}`, `{$col_name}` FROM `{$table_name}` WHERE `{$col_name}` LIKE %s", '%' . $wpdb->esc_like($old_filename) . '%'));
                
                foreach ($rows as $row) {
                    $original_data = $row->$col_name;
                    $primary_key_value = $row->$primary_key;

                    $updated_data = siteblazer_recursive_data_replace($old_filename, $new_url, $original_data);

                    if ($original_data !== $updated_data) {
                        $wpdb->update($table_name, [$col_name => $updated_data], [$primary_key => $primary_key_value]);
                        $changes_were_made = true;
                    }
                }
            }
        }
    }
    return $changes_were_made;
}

function siteblazer_recursive_data_replace($old_filename, $new_full_url, $data) {
    if (is_string($data) && is_serialized($data)) {
        $unpacked_data = @unserialize($data);
        if ($unpacked_data !== false) {
            return serialize(siteblazer_recursive_data_replace($old_filename, $new_full_url, $unpacked_data));
        }
    }
    
    if (is_array($data)) {
        foreach ($data as &$value) {
            $value = siteblazer_recursive_data_replace($old_filename, $new_full_url, $value);
        }
        unset($value);
    } elseif (is_object($data)) {
        $data_vars = get_object_vars($data);
        foreach ($data_vars as $key => &$value) {
            $data->$key = siteblazer_recursive_data_replace($old_filename, $new_full_url, $value);
        }
        unset($value);
    } elseif (is_string($data)) {
        $regex = '/(https?:)?([A-Za-z0-9-_\.\/\\\]*)' . preg_quote($old_filename, '/') . '/';
        $data = preg_replace($regex, $new_full_url, $data);
    }
    
    return $data;
}

function siteblazer_get_pexels_image($api_key, $query, $old_filename = '') {

    // Default size
    $width  = 1200;
    $height = 800;

    // Use exact size if mapped
    if ($old_filename && defined('SITEBLAZER_IMAGE_SIZES') && isset(SITEBLAZER_IMAGE_SIZES[$old_filename])) {
        [$width, $height] = SITEBLAZER_IMAGE_SIZES[$old_filename];
    }

    $url = "https://api.pexels.com/v1/search?query=" . urlencode($query) . "&per_page=1&page=" . rand(1, 800);
    $response = wp_remote_get($url, [
        'headers' => ['Authorization' => $api_key],
        'timeout' => 20
    ]);

    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
        return false;
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);

    if (!isset($data['photos'][0]['src']['original'])) {
        return false;
    }

    // 🔥 IMPORTANT: get exact size from Pexels
    return add_query_arg([
        'w'   => $width,
        'h'   => $height,
        'fit' => 'crop'
    ], $data['photos'][0]['src']['original']);
}


function siteblazer_download_image($image_url, $old_filename = '') {
    require_once ABSPATH . 'wp-admin/includes/file.php';
    require_once ABSPATH . 'wp-admin/includes/media.php';
    require_once ABSPATH . 'wp-admin/includes/image.php';

    $temp_file = download_url($image_url, 30);
    if (is_wp_error($temp_file)) return false;

    
    $file_info = [
        'name'     => basename($old_filename ?: parse_url($image_url, PHP_URL_PATH)),
        'tmp_name'=> $temp_file
    ];

    $attachment_id = media_handle_sideload($file_info, 0);

    if (is_wp_error($attachment_id)) {
        @unlink($temp_file);
        return false;
    }

    return [
        'id'  => $attachment_id,
        'url' => wp_get_attachment_url($attachment_id)
    ];
}

function siteblazer_replace_category_terms_once() {

    if (get_option('siteblazer_category_done')) {
        return;
    }

    $json = json_placeholder_get_json_data();
    if (!$json || empty($json['Category'])) return;

    $terms = get_terms([
        'taxonomy'   => 'product_cat',
        'hide_empty' => false,
    ]);

    foreach ($terms as $term) {

        $description = $term->description;

        if (strpos($description, '{') === false) continue;

        $new_desc = json_placeholder_replace_text($description, $json);

        if ($new_desc !== $description) {
            wp_update_term($term->term_id, 'product_cat', [
                'description' => $new_desc,
            ]);
        }
    }

    update_option('siteblazer_category_done', 1);
}

function siteblazer_console_log($data) {
    if (is_array($data) || is_object($data)) {
        $data = json_encode($data);
    } else {
        $data = json_encode([$data]);
    }
    echo "<script>console.log(" . $data . ");</script>";
}
